<?php

namespace App\Http\Controllers;

use App\Exports\UsersExport;
use App\Imports\UsersImport;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Carbon\Carbon;

class UserController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        // $this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index']]);
        // $this->middleware('permission:user-create', ['only' => ['create','store', 'updateStatus']]);
        // $this->middleware('permission:user-edit', ['only' => ['edit','update']]);
        // $this->middleware('permission:user-delete', ['only' => ['delete']]);
    }


    /**
     * List User
     * @param Nill
     * @return Array $user
     * @author Shani Singh
     */
    public function index()
    {
        $loggedInUser = Auth::user();

        $query = User::select([
                'users.*',
                'roles.role_name as role_name',
                'team_leader.first_name as team_first_name',
                'team_leader.last_name as team_last_name'
            ])
            ->where('users.id', '!=', 1)
            ->leftJoin('roles', 'roles.id', '=', 'users.role_id')
            ->leftJoin('users as team_leader', 'team_leader.id', '=', 'users.team_leader_id');
            
        if($loggedInUser && $loggedInUser->id == 1)
        {
            $query = $query;
        }
        elseif ($loggedInUser && $loggedInUser->role_id == 44) 
        {
            $query->where('users.team_leader_id', '=', $loggedInUser->id);
        }
        else
        {
            $query->where('users.id', '=', $loggedInUser->id);
        }
        
        $query->orWhere('users.id', '=', $loggedInUser->id);
    
        $users = $query->paginate(10);
            
        return view('users.index', ['users' => $users]);
    }
    
    public function fetchExistingCodes() {
        $codes = DB::table('users')->pluck('fpo_employee_code')->filter(function ($value) {
            return !is_null($value) && trim($value) !== '';
        })->values()->toArray();
        return response()->json($codes);
    }
    
    public function view(Request $request, $id){
        $user = User::select([
            'users.*',
            'states.state_name as state_name',
            'districts.district_name as district_name'
        ])
        ->leftJoin('states','states.id','=','users.state_id')
        ->leftJoin('districts','districts.id','=','users.district_id')
        ->findOrFail($id);
        return view('users.view', ['user' => $user]);
    }

    /**
     * Create User
     * @param Nill
     * @return Array $user
     * @author Shani Singh
     */
    public function create()
    {
        // $userId = Auth::user()->id;
        // where('id', '!=', $userId)->
        // $roles = Role::all();
        $roles = Role::where('status','1')->get();
        $users = User::where('status','1')->where('role_id','44')->get();
        $states = DB::table('states')->get();
        $districts = DB::table('districts')->get();
        return view('users.add',['roles' => $roles,'users' => $users,'states'=> $states,'districts' => $districts]);
    }
    
    public function getDistrict($state)
    {
    	$districts = DB::table('districts')->where('state_id', $state)->get();
        $html = view(
    		"users/getDistrict", 
    		[
    			'districts' => $districts
    		]
    	)->render();

    	return Response()->json([
	    	'status' => 'success',
	    	'html' => $html
	    ]);
    }
    
    
   
    
    
    
    

  public function getFpo($fpo)
    {
    $resp = User::select([
            'users.id',
            'users.state_id',
            'users.district_id',
            'users.village',
            'users.block',
            'states.state_name as state_name',
            'districts.district_name as district_name'
        ])
        ->leftJoin('states', 'states.id', '=', 'users.state_id')
        ->leftJoin('districts', 'districts.id', '=', 'users.district_id')
        ->where('team_leader_id', $fpo)
        ->get();
        $responseData = [];
    foreach ($resp as $item) {
        $responseData[] = [
            'state_id' => $item->state_id,
            'state_name' => $item->state_name,
            'district_id' => $item->district_id,
            'district_name' => $item->district_name,
            'block' => $item->block,
            'village' => $item->village
        ];
    }
    return response()->json([
        'status' => 'success',
        'data' => $responseData
    ]);
    }   


// public function getFpo($fpo)
// {
//     $resp = User::select([
//             'users.state_id',
//             'users.district_id',
//             'users.village',
//             'users.block',
//             'states.state_name as state_name',
//             'districts.district_name as district_name'
//         ])
//         ->leftJoin('states', 'states.id', '=', 'users.state_id')
//         ->leftJoin('districts', 'districts.id', '=', 'users.district_id')
//         ->where('team_leader_id', $fpo)
//         ->groupBy('users.state_id', 'users.district_id', 'states.state_name', 'districts.district_name', 'users.village', 'users.block')
//         ->get();

//     $responseData = [];
//     foreach ($resp as $item) {
//         $responseData[] = [
//             'state_id' => $item->state_id,
//             'state_name' => $item->state_name,
//             'district_id' => $item->district_id,
//             'district_name' => $item->district_name,
//             'block' => $item->block,
//             'village' => $item->village
//         ];
//     }

//     return response()->json([
//         'status' => 'success',
//         'data' => $responseData
//     ]);
// }



    

    /**
     * Store User
     * @param Request $request
     * @return View Users
     * @author Shani Singh
     */
    public function store(Request $request)
    {
        $userData = Auth::user(); 
        
        $request->validate([
            // 'first_name'    => 'required',
            // 'last_name'     => 'required',
            // 'email'         => 'required|unique:users,email',
            'email' => 'nullable|email|unique:users,email',
            // 'mobile_number' => 'required|numeric|digits:10',
            // 'status'       =>  'required|numeric|in:0,1',
        ]);

        DB::beginTransaction();
        try {
            if (!$request->has('team_leader_id'))
            {
                $request->team_leader_id = $userData->id;
            }
            // Store Data
            $roleId = $request->role_id;
            unset($request->confirm_password);
            if(isset($roleId) && $roleId == 41 || $roleId == 47 || $roleId == 48)
            {
                $user = User::create([
                    'first_name'    => $request->first_name,
                    'last_name'     => $request->last_name,
                    'team_leader_id'     => $request->team_leader_id,
                    'image'     => $request->image,
                    'email'         => $request->email,
                    'education'      => $request->farmer_education,
                    'education_certificate' => $request->ceo_education_certificate,
                    'mobile_number' => $request->mobile_number,
                    'role_id'       => $roleId,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'husband_father_name' => $request->husband_father_name,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address'   => $request->address,
                    'village'=> $request->village,
                    'dob'   => $request->dob,
                    'occupation' => $request->occupation,
                    'designation' => $request->designation,
                    'doa'   => $request->doa,
                    'aadhaar_number' => $request->aadhaar_number,
                    'pan_number'    => $request->pan_number,
                    'aadhaar_card' => $request->farmer_aadhaar_card,
                    'pan_card' => $request->farmer_pan_card,
                    'gender'    => $request->gender
                ]);    
            }
            // fpo
            elseif(isset($roleId) && $roleId == 45)
            {
                $user = User::create([
                    'first_name'    => $request->bod_first_name,
                    'last_name'    => $request->last_name,
                    'team_leader_id'     => $request->team_leader_id,
                    'image'    => $request->image,
                    'husband_father_name' => $request->bod_husband_father_name,
                    'email'         => $request->email,
                    'education'     => $request->bod_education,
                    'education_certificate' => $request->bod_education_certificate,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'mobile_number' => $request->bod_mobile_number,
                    'role_id'       => $roleId,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address'   => $request->address,
                    'village'=> $request->village,
                    'dob'   => $request->bod_dob,
                    'din_number' => $request->din_number,
                    'occupation' => $request->bod_occupation,
                    'designation' => $request->bod_designation,
                    'doa'   => $request->bod_doa,
                    'toa'   => $request->toa,
                    'num_of_shares'   => $request->num_of_shares,
                    'cls_of_shares' => $request->cls_of_shares,
                    'aadhaar_number' => $request->bod_aadhaar_number,
                    'pan_number'    => $request->bod_pan_number,
                    'divident'  => $request->divident,
                    'other_directorship' => $request->other_directorship,
                    'company_debt' => $request->company_debt,
                    'gender' => $request->bod_gender,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            elseif(isset($roleId) && $roleId == 44){
                
                $user = User::create([
                    // 'image'    => $request->image,
                    'first_name' => $request->fpo_first_name,
                    'role_id'       => $roleId,
                    'team_leader_id'     => $request->team_leader_id,
                    'email'         =>$request->email,
                    'fpo_employee_code' => $request->fpo_employee_code,
                    // 'education'      => $request->education,
                    // 'education_certificate' => $request->education_certificate,
                    'roc_certificate'      => $request->roc_certificate,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'single_share_values' => $request->fpo_single_share_values,
                    'total_share_values' => $request->fpo_total_share_values,
                    'cin_number' => $request->cin_number,
                    'company_logo' => $request->company_logo,
                    'registration_date' => $request->registration_date,
                    'authorized_person_name' => $request->authorized_person_name,
                    'authorized_person_email' => $request->authorized_person_email,
                    'authorized_person_mobile_num' => $request->authorized_person_mobile_num,
                    'authorized_person_designation' => $request->authorized_person_designation,
                    'authorized_person_pan_num' => $request->authorized_person_pan_num,
                    'authorized_person_aadhaar_num' => $request->authorized_person_aadhaar_num,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village'=> $request->village,
                    'pincode' => $request->pincode,
                    'pan_card' => $request->pan_card,
                    'aadhaar_card' => $request->aadhaar_card,
                    'moa' => $request->moa,
                    'aoa' => $request->aoa,
                    'gst_certificate' => $request->gst_certificate,
                    'gst_statement' => $request->gst_statement,
                    'audited_financial' => $request->audited_financial,
                    'bank_statement' => $request->bank_statement,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            elseif(isset($roleId) && $roleId == 46){
                $user = User::create([
                    'first_name'    => $request->farmer_first_name,
                    'last_name'    => $request->farmer_last_name,
                    'team_leader_id'     => $request->team_leader_id,
                    'farmer_code_id' => $request->farmer_code_id,
                    'education'     => $request->farmer_education,
                    'education_certificate' => $request->farmer_education_certificate,
                    'email'    => $request->email,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'image'    => $request->image,
                    'photo'    => $request->photo,
                    'role_id'       => $roleId,
                    'mobile_number' => $request->farmer_mobile_number,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'village' => $request->village,
                    'block' => $request->block,
                    'address' => $request->address,
                    'pincode' => $request->pincode,
                    'aadhaar_number' => $request->farmer_aadhaar_number,
                    'pan_number' => $request->farmer_pan_number,
                    'phone_type' => $request->phone_type,
                    'aadhaar_card' => $request->farmer_aadhaar_card,
                    'pan_card' => $request->farmer_pan_card,
                    'bank_account_number' => $request->bank_account_number,
                    'confirm_bank_account_number' => $request->confirm_bank_account_number,
                    'ifsc_code' => $request->ifsc_code,
                    'num_of_shares' => $request->fpo_num_of_shares,
                    'single_share_values' => $request->fpo_single_share_values,
                    'total_share_values' => $request->fpo_total_share_values,
                    'husband_father_name' => $request->farmer_father_mother_name,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            else
            {
                $user = User::create([
                    'employee_code' => $request->employee_code,
                    'first_name'    => $request->first_name,
                    'last_name'     => $request->last_name, 
                    'team_leader_id'     => $request->team_leader_id,
                    'email'         => $request->email,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password' => bcrypt($request->confirm_password),
                    'mobile_number' => $request->mobile_number,
                    'role_id'       => $roleId, 
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village'=> $request->village,
                    'pincode' => $request->pincode,
                    'team_leader_id'=> $request->team_leader_id,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '_' . $image->getClientOriginalName();
                $image->storeAs('public/images', $imageName); // Store the image in storage/app/public/images
            }
            
            if ($request->hasFile('education_certificate')) {
                $educationCertificate = $request->file('education_certificate');
                $educationCertificateName = time() . '_' . $educationCertificate->getClientOriginalName();
                $educationCertificate->storeAs('public/images', $educationCertificateName);
            }
            
            if ($request->hasFile('roc_certificate')) {
                $roc_certificate = $request->file('roc_certificate');
                $roc_certificateName = time() . '_' . $roc_certificate->getClientOriginalName();
                $roc_certificate->storeAs('public/images', $roc_certificateName);
            }
            
            if ($request->hasFile('pan_card')) {
                $pan_card = $request->file('pan_card');
                $pan_cardName = time() . '_' . $pan_card->getClientOriginalName();
                $pan_card->storeAs('public/images', $pan_cardName);
            }
            
            if ($request->hasFile('aadhaar_card')) {
                $aadhaar_card = $request->file('aadhaar_card');
                $aadhaar_cardName = time() . '_' . $aadhaar_card->getClientOriginalName();
                $aadhaar_card->storeAs('public/images', $aadhaar_cardName);
            }
            
            if ($request->hasFile('moa')) {
                $moa = $request->file('moa');
                $moaName = time() . '_' . $moa->getClientOriginalName();
                $moa->storeAs('public/images', $moaName);
            }
            
            if ($request->hasFile('aoa')) {
                $aoa = $request->file('aoa');
                $aoaName = time() . '_' . $aoa->getClientOriginalName();
                $aoa->storeAs('public/images', $aoaName);
            }
            
            if ($request->hasFile('gst_certificate')) {
                $gst_certificate = $request->file('gst_certificate');
                $gst_certificateName = time() . '_' . $gst_certificate->getClientOriginalName();
                $gst_certificate->storeAs('public/images', $gst_certificateName);
            }
            
            if ($request->hasFile('gst_statement')) {
                $gst_statement = $request->file('gst_statement');
                $gst_statementName = time() . '_' . $gst_statement->getClientOriginalName();
                $gst_statement->storeAs('public/images', $gst_statementName);
            }
            
            if ($request->hasFile('audited_financial')) {
                $audited_financial = $request->file('audited_financial');
                $audited_financialName = time() . '_' . $audited_financial->getClientOriginalName();
                $audited_financial->storeAs('public/images', $audited_financialName);
            }
            
            if ($request->hasFile('bank_statement')) {
                $bank_statement = $request->file('bank_statement');
                $bank_statementName = time() . '_' . $bank_statement->getClientOriginalName();
                $bank_statement->storeAs('public/images', $bank_statementName);
            }

            // Delete Any Existing Role
            // DB::table('model_has_roles')->where('model_id',$user->id)->delete();

            // Assign Role To User
            // $user->assignRole($user->role_id);

            // Commit And Redirected To Listing
            DB::commit();
            return redirect()->route('users.index')->with('success','User Created Successfully.');

        } catch (\Throwable $th) {
            // Rollback and return with Error
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }

    /**
     * Update Status Of User
     * @param Integer $status
     * @return List Page With Success
     * @author Shani Singh
     */
    public function updateStatus($user_id, $status)
    {
        // Validation
        $validate = Validator::make([
            'user_id'   => $user_id,
            'status'    => $status
        ], [
            'user_id'   =>  'required|exists:users,id',
            'status'    =>  'required|in:0,1',
        ]);

        // If Validations Fails
        if($validate->fails()){
            return redirect()->route('users.index')->with('error', $validate->errors()->first());
        }

        try {
            DB::beginTransaction();

            // Update Status
            User::whereId($user_id)->update(['status' => $status]);

            // Commit And Redirect on index with Success Message
            DB::commit();
            return redirect()->route('users.index')->with('success','User Status Updated Successfully!');
        } catch (\Throwable $th) {

            // Rollback & Return Error Message
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Edit User
     * @param Integer $user
     * @return Collection $user
     * @author Shani Singh
     */
    public function edit(User $user)
    {
        $roles = Role::where('status','1')->get();
        // $users = User::where('id', '!=', $user->id)->where('status','1')->get();
        $users = User::where('status','1')->where('role_id','44')->get();
        $states = DB::table('states')->get();
        $districts = DB::table('districts')->where('state_id',$user->state_id)->get();
        return view('users.edit',['roles' => $roles,'users' => $users,'states' => $states, 'districts' => $districts])->with([

            'user'  => $user
        ]);
    }

    /**
     * Update User
     * @param Request $request, User $user
     * @return View Users
     * @author Shani Singh
     */
    public function update(Request $request, User $user)
    {
        // Validations
        $request->validate([
            // 'first_name'    => 'required',
            // 'last_name'     => 'required',
            'email'         => 'nullable|unique:users,email,'.$user->id.',id',
            // 'mobile_number' => 'required|numeric|digits:10',
            // 'status'       =>  'required|numeric|in:0,1',
        ]);
        
        $imageName = '';
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = isset($image) && $image ? time() . '_' . $image->getClientOriginalName() : null;
            $image->storeAs('public/images', $imageName); // Store the image in storage/app/public/images
        }

        DB::beginTransaction();
        try {

            // Store Data
            // $user_updated = User::whereId($user->id)->update([
            //     'employee_code' => $request->employee_code,
            //     'first_name'    => $request->first_name,
            //     'last_name'     => $request->last_name,
            //     'email'         => $request->email,
            //     'mobile_number' => $request->mobile_number,
            //     'role_id'       => $request->role_id,
            //     'team_leader_id'=> $request->team_leader_id,
            //     'status'        => $request->status,
            //     'password'      => bcrypt($request->password),
            // ]);
            
            $roleId = $request->role_id;
            unset($request->confirm_password);
            if(isset($roleId) && $roleId == 41 || $roleId == 47 || $roleId == 48)
            {
                $user_updated = User::whereId($user->id)->update([
                    'first_name'    => $request->first_name,
                    'last_name'     => $request->last_name,
                    'team_leader_id'     => $request->team_leader_id,
                    'image'     => $imageName,
                    'email'         => $request->email,
                    'education'      => $request->ceo_education,
                    'education_certificate' => $request->ceo_education_certificate,
                    'mobile_number' => $request->mobile_number,
                    'role_id'       => $roleId,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'husband_father_name' => $request->husband_father_name,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village' => $request->village,
                    'pincode' => $request->pincode,
                    'dob'   => $request->dob,
                    'occupation' => $request->occupation,
                    'designation' => $request->designation,
                    'doa'   => $request->doa,
                    'aadhaar_number' => $request->aadhaar_number,
                    'pan_number'    => $request->pan_number,
                    'gender'    => $request->gender
                ]);  
            }
            // fpo
            elseif(isset($roleId) && $roleId == 45)
            {
                $user_updated = User::whereId($user->id)->update([
                    'first_name'    => $request->bod_first_name,
                    'last_name'    => $request->last_name,
                    'team_leader_id'     => $request->team_leader_id,
                    'image'    => $imageName,
                    'husband_father_name' => $request->bod_husband_father_name,
                    'email'         => $request->email,
                    'education'     => $request->bod_education,
                    'education_certificate' => $request->bod_education_certificate,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'mobile_number' => $request->bod_mobile_number,
                    'role_id'       => $roleId,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village' => $request->village,
                    'pincode' => $request->pincode,
                    'dob'   => $request->bod_dob,
                    'din_number' => $request->din_number,
                    'occupation' => $request->bod_occupation,
                    'designation' => $request->bod_designation,
                    'doa'   => $request->bod_doa,
                    'toa'   => $request->toa,
                    'num_of_shares'   => $request->num_of_shares,
                    'cls_of_shares' => $request->cls_of_shares,
                    'aadhaar_number' => $request->bod_aadhaar_number,
                    'pan_number'    => $request->bod_pan_number,
                    'divident'   => $request->divident,
                    'other_directorship' => $request->other_directorship,
                    'company_debt' => $request->company_debt,
                    'gender' => $request->bod_gender,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            elseif(isset($roleId) && $roleId == 44)
            {
                $user_updated = User::whereId($user->id)->update([
                    // 'image'    => $imageName,
                    'first_name' => $request->fpo_first_name,
                    'role_id'       => $roleId,
                    'team_leader_id'     => $request->team_leader_id,
                    'email'         =>$request->email,
                    'fpo_employee_code' => $request->fpo_employee_code,
                    // 'education'      => $request->education,
                    // 'education_certificate' => $request->education_certificate,
                    'roc_certificate'      => $request->roc_certificate,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'single_share_values' => $request->fpo_single_share_values,
                    'total_share_values' => $request->fpo_total_share_values,
                    'cin_number' => $request->cin_number,
                    'company_logo' => $request->company_logo,
                    'registration_date' => $request->registration_date,
                    'authorized_capital_rs' => $request->authorized_capital_rs,
                    'paidup_capital_rs' => $request->paidup_capital_rs,
                    'authorized_person_name' => $request->authorized_person_name,
                    'authorized_person_email' => $request->authorized_person_email,
                    'authorized_person_mobile_num' => $request->authorized_person_mobile_num,
                    'authorized_person_designation' => $request->authorized_person_designation,
                    'authorized_person_pan_num' => $request->authorized_person_pan_num,
                    'authorized_person_aadhaar_num' => $request->authorized_person_aadhaar_num,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village' => $request->village,
                    'pincode' => $request->pincode,
                    'pan_card' => $request->pan_card,
                    'aadhaar_card' => $request->aadhaar_card,
                    'moa' => $request->moa,
                    'aoa' => $request->aoa,
                    'gst_certificate' => $request->gst_certificate,
                    'gst_statement' => $request->gst_statement,
                    'audited_financial' => $request->audited_financial,
                    'bank_statement' => $request->bank_statement,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            elseif(isset($roleId) && $roleId == 46)
            {
                $user_updated = User::whereId($user->id)->update([
                    'first_name'    => $request->farmer_first_name,
                    'last_name'    => $request->farmer_last_name,
                    'team_leader_id' => $request->team_leader_id,
                    'education'     => $request->farmer_education,
                    'education_certificate' => $request->farmer_education_certificate,
                    'email'    => $request->email,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password'      => bcrypt($request->confirm_password),
                    'image'    => $imageName,
                    'photo'    => $request->photo,
                    'role_id'       => $roleId,
                    'mobile_number' => $request->farmer_mobile_number,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'village' => $request->village,
                    'block' => $request->block,
                    'address' => $request->address,
                    'pincode' => $request->pincode,
                    'aadhaar_number' => $request->farmer_aadhaar_number,
                    'pan_number' => $request->farmer_pan_number,
                    'phone_type' => $request->phone_type,
                    'aadhaar_card' => $request->farmer_aadhaar_card,
                    'pan_card' => $request->farmer_pan_card,
                    'bank_account_number' => $request->bank_account_number,
                    'confirm_bank_account_number' => $request->confirm_bank_account_number,
                    'ifsc_code' => $request->ifsc_code,
                    'num_of_shares' => $request->fpo_num_of_shares,
                    'single_share_values' => $request->fpo_single_share_values,
                    'total_share_values' => $request->fpo_total_share_values,
                    'husband_father_name' => $request->farmer_father_mother_name,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            else
            {
                $user_updated = User::whereId($user->id)->update([
                    'employee_code' => $request->employee_code,
                    'first_name'    => $request->first_name,
                    'last_name'     => $request->last_name, 
                    'team_leader_id'     => $request->team_leader_id,
                    'email'         => $request->email,
                    'password'      => bcrypt($request->password),
                    'decrypt_password' => $request->password,
                    // 'confirm_password' => bcrypt($request->confirm_password),
                    'mobile_number' => $request->mobile_number,
                    'state_id' => $request->state_id,
                    'district_id' => $request->district_id,
                    'block' => $request->block,
                    'address' => $request->address,
                    'village' => $request->village,
                    'pincode' => $request->pincode,
                    'role_id'       => $roleId, 
                    'team_leader_id'=> $request->team_leader_id,
                    'status'        => isset($request->status) && $request->status ? $request->status : 1,
                ]);
            }
            
            if ($request->hasFile('education_certificate')) {
                $educationCertificate = $request->file('education_certificate');
                $educationCertificateName = time() . '_' . $educationCertificate->getClientOriginalName();
                $educationCertificate->storeAs('public/images', $educationCertificateName);
            }
            
            if ($request->hasFile('roc_certificate')) {
                $roc_certificate = $request->file('roc_certificate');
                $roc_certificateName = time() . '_' . $roc_certificate->getClientOriginalName();
                $roc_certificate->storeAs('public/images', $roc_certificateName);
            }
            
            if ($request->hasFile('pan_card')) {
                $pan_card = $request->file('pan_card');
                $pan_cardName = time() . '_' . $pan_card->getClientOriginalName();
                $pan_card->storeAs('public/images', $pan_cardName);
            }
            
            if ($request->hasFile('aadhaar_card')) {
                $aadhaar_card = $request->file('aadhaar_card');
                $aadhaar_cardName = time() . '_' . $aadhaar_card->getClientOriginalName();
                $aadhaar_card->storeAs('public/images', $aadhaar_cardName);
            }
            
            if ($request->hasFile('moa')) {
                $moa = $request->file('moa');
                $moaName = time() . '_' . $moa->getClientOriginalName();
                $moa->storeAs('public/images', $moaName);
            }
            
            if ($request->hasFile('aoa')) {
                $aoa = $request->file('aoa');
                $aoaName = time() . '_' . $aoa->getClientOriginalName();
                $aoa->storeAs('public/images', $aoaName);
            }
            
            if ($request->hasFile('gst_certificate')) {
                $gst_certificate = $request->file('gst_certificate');
                $gst_certificateName = time() . '_' . $gst_certificate->getClientOriginalName();
                $gst_certificate->storeAs('public/images', $gst_certificateName);
            }
            
            if ($request->hasFile('gst_statement')) {
                $gst_statement = $request->file('gst_statement');
                $gst_statementName = time() . '_' . $gst_statement->getClientOriginalName();
                $gst_statement->storeAs('public/images', $gst_statementName);
            }
            
            if ($request->hasFile('audited_financial')) {
                $audited_financial = $request->file('audited_financial');
                $audited_financialName = time() . '_' . $audited_financial->getClientOriginalName();
                $audited_financial->storeAs('public/images', $audited_financialName);
            }
            
            if ($request->hasFile('bank_statement')) {
                $bank_statement = $request->file('bank_statement');
                $bank_statementName = time() . '_' . $bank_statement->getClientOriginalName();
                $bank_statement->storeAs('public/images', $bank_statementName);
            }
            
            // Delete Any Existing Role
            // DB::table('model_has_roles')->where('model_id',$user->id)->delete();

            // Assign Role To User
            // $user->assignRole($user->role_id);

            // Commit And Redirected To Listing
            DB::commit();
            return redirect()->route('users.index')->with('success','User Updated Successfully.');

        } catch (\Throwable $th) {
            // Rollback and return with Error
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }

    /**
     * Delete User
     * @param User $user
     * @return Index Users
     * @author Shani Singh
     */
    public function delete(User $user)
    {

        DB::beginTransaction();
        try {
            // Delete User
            User::whereId($user->id)->delete();

            DB::commit();
            return redirect()->route('users.index')->with('success', 'User Deleted Successfully!.');

        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Import Users
     * @param Null
     * @return View File
     */
    public function importUsers()
    {
        return view('users.import');
    }
    
    public function uploadUsers(Request $request)
    {
        
        $request->validate([
            'file' => 'required|file|mimes:csv,txt',
        ]);
    
        $file = $request->file('file');
    
        if (($handle = fopen($file->getPathname(), 'r')) === false) {
            return redirect()->back()->with('error', 'Error opening file.');
        }
    
        $header = fgetcsv($handle);
    
        if ($header === false) {
            fclose($handle);
            return redirect()->back()->with('error', 'Error reading file header.');
        }
    
        $existingFarmerCodes = User::pluck('farmer_code_id')->filter()->values()->toArray();
    
        $codesByBase = [];
        foreach ($existingFarmerCodes as $code) {
            $parts = explode('-', $code);
            $baseCode = implode('-', array_slice($parts, 0, 2));
            $suffix = isset($parts[2]) ? $parts[2] : '001'; 
    
            if (!isset($codesByBase[$baseCode])) {
                $codesByBase[$baseCode] = [];
            }
    
            $codesByBase[$baseCode][] = $suffix;
        }
        
        $rowCount = 0;
        $fpoEmployeeCodes = [];
    
        while (($rowData = fgetcsv($handle)) !== false)
        {
            $rowCount++;
            $validator = Validator::make([
                'fpo_name' => $rowData[0],
                'fpo_employee_code' => $rowData[1],
                'email' => isset($rowData[2]) ? $rowData[2] : null,
                'aadhaar_number' => isset($rowData[11]) ? $rowData[11] : null,
                'pan_number' => isset($rowData[12]) ? $rowData[12] : null,
                'state' => $rowData[4],
                'district' => $rowData[5],
                'block' => $rowData[6],
                'address' => $rowData[7],
                'first_name' => $rowData[8],
                'husband_father_name' => $rowData[10],
                'mobile_number' => $rowData[11],
            ], [
                'fpo_name' => 'required|string|max:255',
                'fpo_employee_code' => 'required', 
                'first_name' => 'required', 
                'husband_father_name' => 'required',
                'email' => 'nullable|email|unique:users,email',
                'mobile_number' => 'required',
                'aadhaar_number' => 'required',
                'pan_number' => 'nullable',
                'state' => 'required',
                'district' => 'required',
                'block' => 'required',
                'address' => 'required',
            ]);
    
            if ($validator->fails()) 
            {
                fclose($handle);
                return redirect()->back()->with('error', 'Validation error on row ' . $rowCount . ': ' . implode(', ', $validator->errors()->all()));
            }
            
            
            
    
            $fpoEmployeeCode = $rowData[1];
            $codeParts = explode('-', $fpoEmployeeCode);
            $baseCode = implode('-', array_slice($codeParts, 0, 2));
           
            $newSuffix = '001';
            
            // Check if suffix for baseCode has already been used
           
            if (isset($codesByBase[$baseCode])) {
                $existingSuffixes = $codesByBase[$baseCode];
                $maxSuffix = max(array_map('intval', $existingSuffixes));
                $newSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
            }
            
            // Ensure the newSuffix is unique
            $uniqueSuffix = $newSuffix;
            while (in_array($baseCode . '-' . $uniqueSuffix, $existingFarmerCodes) || in_array($uniqueSuffix, $fpoEmployeeCodes[$fpoEmployeeCode] ?? [])) {
                $maxSuffix++;
                $uniqueSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
            }
             
            // Add suffix to tracking for current fpo_employee_code
            $fpoEmployeeCodes[$fpoEmployeeCode][] = $uniqueSuffix;
    
            $newFarmerCode = $baseCode . '-' . $uniqueSuffix;
    
            $teamLeader = User::where('fpo_employee_code', $fpoEmployeeCode)->first();
            $teamLeaderId = $teamLeader ? $teamLeader->id : null;
            
            $states = DB::table('states')->get();
            $stateName = $rowData[4];
            $state = $states->firstWhere('state_name', $stateName);
            $stateId = $state ? $state->id : null;
            
            $districts = DB::table('districts')->get();
            $districtName = $rowData[5];
            $district = $districts->firstWhere('district_name', $districtName);
            $districtId = $district ? $district->id : null;
            
            unset($rowData[12]);
            try {
                User::updateOrCreate(
                    [
                        'role_id' => 46,
                        'fpo_name' => $rowData[0],
                        'email' => $rowData[2],
                        'farmer_code_id' => $newFarmerCode,
                        'password' => bcrypt($rowData[3]),
                        'decrypt_password' => $rowData[3],
                        'state_id' => $stateId,
                        'district_id' => $districtId,
                        'block' => $rowData[6],
                        'address' => $rowData[7],
                        'first_name' => $rowData[8],
                        'last_name' => $rowData[9],
                        'husband_father_name' => $rowData[10],
                        'mobile_number' => $rowData[11],
                        // 'aadhaar_number' => $rowData[12],
                        'pan_number' => $rowData[13],
                        'phone_type' => $rowData[14],
                        'bank_account_number' => $rowData[15],
                        'ifsc_code' => $rowData[16],
                        'num_of_shares' => $rowData[17],
                        'village' => $rowData[18],
                        'single_share_values' => $rowData[19],
                        'total_share_values' => $rowData[20],
                        'team_leader_id' => $teamLeaderId,
                        'created_at' => Carbon::now()->toDateTimeString(),
                        'updated_at' => Carbon::now()->toDateTimeString(),
                    ]
                );
            } catch (\Exception $e) {
                fclose($handle);
                return redirect()->back()->with('error', 'Error processing row ' . $rowCount . ': ' . $e->getMessage());
            }
        }
    
        fclose($handle); 
    
        if ($rowCount === 0) {
            return redirect()->back()->with('error', 'No data rows found in the file.');
        }
    
        return redirect()->back()->with('success', 'Users imported successfully!');
    }

    
    // public function uploadUsers(Request $request)
    // {
    //     $request->validate([
    //         'file' => 'required|file|mimes:csv,txt',
    //     ]);
    
    //     $file = $request->file('file');
        
    //     if (($handle = fopen($file->getPathname(), 'r')) === false) {
    //         return redirect()->back()->with('error', 'Error opening file.');
    //     }
        
    //     $header = fgetcsv($handle);
        
    //     if ($header === false) {
    //         fclose($handle);
    //         return redirect()->back()->with('error', 'Error reading file header.');
    //     }
        
    //     $existingFarmerCodes = User::pluck('farmer_code_id')->filter()->values()->toArray();

    //     $codesByBase = [];
    //     foreach ($existingFarmerCodes as $code) {
    //         $parts = explode('-', $code);
    //         $baseCode = implode('-', array_slice($parts, 0, 2));
    //         $suffix = isset($parts[2]) ? $parts[2] : '001'; 
    
    //         if (!isset($codesByBase[$baseCode])) {
    //             $codesByBase[$baseCode] = [];
    //         }
    
    //         $codesByBase[$baseCode][] = $suffix;
    //     }
        
        
    //     $rowCount = 0;
    //     while (($rowData = fgetcsv($handle)) !== false) 
    //     {
    //         $rowCount++;   
    //         $validator = Validator::make([
    //             'fpo_name' => $rowData[0],
    //             'fpo_employee_code' => $rowData[1],
    //             'email' => isset($rowData[2]) ? $rowData[2] : null,
    //             'aadhaar_number' => isset($rowData[11]) ? $rowData[11] : null,
    //             'pan_number' => isset($rowData[12]) ? $rowData[12] : null,
    //             'state' => $rowData[4],
    //             'district' => $rowData[5],
    //             'block' => $rowData[6],
    //             'address' => $rowData[7],
    //             'first_name' => $rowData[8],
    //             'mobile_number' => $rowData[10],
    //             'husband_father_name' => $rowData[19],
    //         ], [
    //             'fpo_name' => 'required|string|max:255',
    //             'fpo_employee_code' => 'required', 
    //             'email' => 'nullable|email|unique:users,email',
    //             'aadhaar_number' => 'nullable|numeric|digits:12',
    //             'pan_number' => 'nullable|alpha_num|size:10',
    //             'state' => 'required',
    //             'district' => 'required',
    //             'block' => 'required',
    //             'address' => 'required',
    //             'first_name' => 'required', 
    //             'mobile_number' => 'required',
    //             'husband_father_name' => 'required'
    //         ]);
            
    //         if ($validator->fails()) 
    //         {
    //             fclose($handle);
    //             return redirect()->back()->with('error', 'Validation error on row ' . $rowCount . ': ' . implode(', ', $validator->errors()->all()));
    //         }
            
    //         $originalFarmerCode = $rowData[1];
    //         $codeParts = explode('-', $originalFarmerCode);
    //         $baseCode = implode('-', array_slice($codeParts, 0, 2)); 
    //         $newSuffix = '001'; 
    
    //         if (isset($codesByBase[$baseCode])) {
    //             $existingSuffixes = $codesByBase[$baseCode];
    //             $maxSuffix = max(array_map('intval', $existingSuffixes));
    //             $newSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
    //         }
    
    //         $newFarmerCode = $baseCode . '-' . $newSuffix;
            
    //         while (in_array($newFarmerCode, $existingFarmerCodes)) {
    //             $maxSuffix++;
    //             $newSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
    //             $newFarmerCode = $baseCode . '-' . $newSuffix;
    //         }
            
    //         $teamLeader = User::where('fpo_employee_code', $rowData[1])->first();
    //         $teamLeaderId = $teamLeader ? $teamLeader->id : null;
            
    //         try {
    //             User::updateOrCreate(
    //                 [
    //                     'role_id' => 46,
    //                     'fpo_name' => $rowData[0],
    //                     // 'team_leader_id' => $rowData[1],
    //                     'email' => $rowData[2],
    //                     'farmer_code_id' => $newFarmerCode,
    //                     'password' => bcrypt($rowData[3]),
    //                     'decrypt_password' => $rowData[3],
    //                     'state' => $rowData[4],
    //                     'district' => $rowData[5],
    //                     'block' => $rowData[6],
    //                     'address' => $rowData[7],
    //                     'first_name' => $rowData[8],
    //                     'last_name' => $rowData[9],
    //                     'mobile_number' => $rowData[10],
    //                     'aadhaar_number' => $rowData[11],
    //                     'pan_number' => $rowData[12],
    //                     'phone_type' => $rowData[13],
    //                     'bank_account_number' => $rowData[14],
    //                     'ifsc_code' => $rowData[15],
    //                     'num_of_shares' => $rowData[16],
    //                     'single_share_values' => $rowData[17],
    //                     'total_share_values' => $rowData[18],
    //                     'husband_father_name' => $rowData[19],
    //                     'team_leader_id' => $teamLeaderId,
    //                     'created_at' => Carbon::now()->toDateTimeString(),
    //                     'updated_at' => Carbon::now()->toDateTimeString(),
    //                 ]
    //             );
    //         } catch (\Exception $e) {
    //             fclose($handle);
    //             return redirect()->back()->with('error', 'Error processing row ' . $rowCount . ': ' . $e->getMessage());
    //         }
    //     }
    
    //     fclose($handle); 
    
    //     if ($rowCount === 0) {
    //         return redirect()->back()->with('error', 'No data rows found in the file.');
    //     }
    
    //     return redirect()->back()->with('success', 'Users imported successfully!');
    // }
    
    // public function uploadUsers(Request $request)
    // {
    //     $request->validate([
    //         'file' => 'required|file|mimes:csv,txt',
    //     ]);
    
    //     $file = $request->file('file');
    //     if (($handle = fopen($file->getPathname(), 'r')) === false) {
    //         return redirect()->back()->with('error', 'Error opening file.');
    //     }
    
    //     $header = fgetcsv($handle);
    
    //     if ($header === false) {
    //         fclose($handle);
    //         return redirect()->back()->with('error', 'Error reading file header.');
    //     }
    
    //     $existingFarmerCodes = User::pluck('farmer_code_id')->filter()->values()->toArray();
    
    //     $codesByBase = [];
    //     foreach ($existingFarmerCodes as $code) {
    //         $parts = explode('-', $code);
    //         $baseCode = implode('-', array_slice($parts, 0, 2));
    //         $suffix = isset($parts[2]) ? $parts[2] : '001'; 
    
    //         if (!isset($codesByBase[$baseCode])) {
    //             $codesByBase[$baseCode] = [];
    //         }
    
    //         $codesByBase[$baseCode][] = $suffix;
    //     }
    
    //     $rowCount = 0;
    //     $fpoEmployeeCodes = [];
    
    //     while (($rowData = fgetcsv($handle)) !== false)
    //     {
    //         $rowCount++;
    //         $validator = Validator::make([
    //             'fpo_name' => $rowData[0],
    //             'fpo_employee_code' => $rowData[1],
    //             'email' => isset($rowData[2]) ? $rowData[2] : null,
    //             'aadhaar_number' => isset($rowData[11]) ? $rowData[11] : null,
    //             'pan_number' => isset($rowData[12]) ? $rowData[12] : null,
    //             'state' => $rowData[4],
    //             'district' => $rowData[5],
    //             'block' => $rowData[6],
    //             'address' => $rowData[7],
    //             'first_name' => $rowData[8],
    //             'husband_father_name' => $rowData[10],
    //             'mobile_number' => $rowData[11],
    //         ], [
    //             'fpo_name' => 'required|string|max:255',
    //             'fpo_employee_code' => 'required', 
    //             'first_name' => 'required', 
    //             'email' => 'nullable|email|unique:users,email',
    //             'husband_father_name' => 'required',
    //             'aadhaar_number' => 'nullable|numeric|digits:12',
    //             'pan_number' => 'nullable|alpha_num|size:10',
    //             'state' => 'required',
    //             'district' => 'required',
    //             'block' => 'required',
    //             'address' => 'required',
    //             'mobile_number' => 'required',
    //         ]);
    
    //         if ($validator->fails()) 
    //         {
    //             fclose($handle);
    //             return redirect()->back()->with('error', 'Validation error on row ' . $rowCount . ': ' . implode(', ', $validator->errors()->all()));
    //         }
    
    //         $fpoEmployeeCode = $rowData[1];
    //         $codeParts = explode('-', $fpoEmployeeCode);
    //         $baseCode = implode('-', array_slice($codeParts, 0, 2));
    //         $newSuffix = '001';
    
    //         // Check if suffix for baseCode has already been used
    //         if (isset($codesByBase[$baseCode])) {
    //             $existingSuffixes = $codesByBase[$baseCode];
    //             $maxSuffix = max(array_map('intval', $existingSuffixes));
    //             $newSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
    //         }
    
    //         // Ensure the newSuffix is unique
    //         $uniqueSuffix = $newSuffix;
    //         while (in_array($baseCode . '-' . $uniqueSuffix, $existingFarmerCodes) || in_array($uniqueSuffix, $fpoEmployeeCodes[$fpoEmployeeCode] ?? [])) {
    //             $maxSuffix++;
    //             $uniqueSuffix = str_pad($maxSuffix + 1, 3, '0', STR_PAD_LEFT);
    //         }
    
    //         // Add suffix to tracking for current fpo_employee_code
    //         $fpoEmployeeCodes[$fpoEmployeeCode][] = $uniqueSuffix;
    
    //         $newFarmerCode = $baseCode . '-' . $uniqueSuffix;
    
    //         $teamLeader = User::where('fpo_employee_code', $fpoEmployeeCode)->first();
    //         $teamLeaderId = $teamLeader ? $teamLeader->id : null;
    
    //         try {
    //             User::updateOrCreate(
    //                 [
    //                     'role_id' => 46,
    //                     'fpo_name' => $rowData[0],
    //                     'email' => $rowData[2],
    //                     'farmer_code_id' => $newFarmerCode,
    //                     'password' => bcrypt($rowData[3]),
    //                     'decrypt_password' => $rowData[3],
    //                     'state' => $rowData[4],
    //                     'district' => $rowData[5],
    //                     'block' => $rowData[6],
    //                     'address' => $rowData[7],
    //                     'first_name' => $rowData[8],
    //                     'last_name' => $rowData[9],
    //                     'husband_father_name' => $rowData[10],
    //                     'mobile_number' => $rowData[11],
    //                     'aadhaar_number' => $rowData[12],
    //                     'pan_number' => $rowData[13],
    //                     'phone_type' => $rowData[14],
    //                     'bank_account_number' => $rowData[15],
    //                     'ifsc_code' => $rowData[16],
    //                     'num_of_shares' => $rowData[17],
    //                     'single_share_values' => $rowData[18],
    //                     'total_share_values' => $rowData[19],
    //                     'team_leader_id' => $teamLeaderId,
    //                     'created_at' => Carbon::now()->toDateTimeString(),
    //                     'updated_at' => Carbon::now()->toDateTimeString(),
    //                 ]
    //             );
    //         } catch (\Exception $e) {
    //             fclose($handle);
    //             return redirect()->back()->with('error', 'Error processing row ' . $rowCount . ': ' . $e->getMessage());
    //         }
    //     }
    
    //     fclose($handle); 
    
    //     if ($rowCount === 0) {
    //         return redirect()->back()->with('error', 'No data rows found in the file.');
    //     }
    
    //     return redirect()->back()->with('success', 'Users imported successfully!');
    // }
}
