<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Auth;

class RolesController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    
    public function __construct()
    {
        $this->middleware('auth');
        // $this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index']]);
        // $this->middleware('permission:user-create', ['only' => ['create','store', 'updateStatus']]);
        // $this->middleware('permission:user-edit', ['only' => ['edit','update']]);
        // $this->middleware('permission:user-delete', ['only' => ['delete']]);
    }


    /**
     * List User
     * @param Nill
     * @return Array $user
     * @author Rakesh Kumar
     */
    public function index()
    {
        $roles = Role::paginate(10);
        return view('roles.index', ['roles' => $roles]);
    }

    /**
     * Create User
     * @param Nill
     * @return Array $user
     * @author Rakesh Kumar
     */
    public function create()
    {
        $menu_info = DB::table('tbl_menu')->where('status', '1')->get();
        // print_r($menu_info); die;
        $menu = [];
        foreach ($menu_info as $item) {
            $menu['parents'][$item->parent][] = $item;
        }

        $data['result'] = Role::buildChild(0, $menu);
        
        return view('roles.add');
    }

    /**
     * Store User
     * @param Request $request
     * @return View Users
     * @author Rakesh Kumar
     */
    public function store(Request $request)
    {
        $userId = Auth::user()->id;
        
        $request->validate([
            'role_name' => 'required',
            'status' => 'required|numeric|in:0,1',
        ]);
    
        DB::beginTransaction();
        try {
            $role = Role::create([
                'user_id' => isset($userId) && $userId ? $userId : '',
                'role_name' => $request->role_name,
                'app_login' => $request->app_login,
                'anexure_approval' => $request->anexure_approval,
                'status' => $request->status,
            ]);
            
            $menuData = [];
            for ($i = 1; $i <= 26; $i++) {
                $menuData[] = [
                    'role_id' => $role->id,
                    'menu_id' => $i,
                    'view' => $request->input("view_$i", 0),
                    'created' => $request->input("created_$i", 0),
                    'edit' => $request->input("edited_$i", 0),
                    'deleted' => $request->input("deleted_$i", 0),
                ];
            }
    
            // Insert or Update Menu Permissions
            foreach ($menuData as $data) {
                DB::table('tbl_role_permissions')->updateOrInsert(
                    ['role_id' => $data['role_id'], 'menu_id' => $data['menu_id']],
                    $data
                );
            }
    
            DB::commit();
            return redirect()->route('roles.index')->with('success', 'Role Created Successfully.');
    
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }

    /**
     * Update Status Of User
     * @param Integer $status
     * @return List Page With Success
     * @author Rakesh Kumar
     */
    public function updateStatus($role_id, $status)
    {
        // Validation
        $validate = Validator::make([
            'role_id'   => $role_id,
            'status'    => $status
        ], [
            'role_id'   =>  'required|exists:roles,id',
            'status'    =>  'required|in:0,1',
        ]);

        // If Validations Fails
        if($validate->fails()){
            return redirect()->route('roles.index')->with('error', $validate->errors()->first());
        }

        try {
            DB::beginTransaction();

            // Update Status
            Role::whereId($role_id)->update(['status' => $status]);

            // Commit And Redirect on index with Success Message
            DB::commit();
            return redirect()->route('roles.index')->with('success','Role Status Updated Successfully!');
        } catch (\Throwable $th) {

            // Rollback & Return Error Message
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Edit Role
     * @param Integer $role
     * @return Collection $role
     * @author Rakesh Kumar
     */
    public function edit(Role $role)
    {
        $menu_info = DB::table('tbl_menu')->where('status', '1')->get();
       
        $menu = [];
        foreach ($menu_info as $item) {
            $menu['parents'][$item->parent][] = $item;
        }

        $data['result'] = Role::buildChild(0, $menu);
        $roleAndPermissionData = DB::table('tbl_role_permissions')->where('role_id',$role->id)->get();
         
        return view('roles.edit', ['role_permissions'=> $roleAndPermissionData])->with([

            'role'  => $role
        ]);
    }

    /**
     * Update Role
     * @param Request $request, User $user
     * @return View Users
     * @author Rakesh Kumar
     */
    public function update(Request $request, Role $role)
    {
        //dd($request->all());
        // Validations
        $userId = Auth::user()->id;
        $request->validate([
            'role_name' => 'required',
            'status' => 'required|numeric|in:0,1',
        ]);

        DB::beginTransaction();
        try {

            // Store Data
            $user_updated = Role::whereId($role->id)->update([
                'user_id' => isset($userId) && $userId ? $userId : '',
                'role_name' => $request->role_name,
                'app_login' => $request->app_login,
                'anexure_approval' => $request->anexure_approval,
                'status' => $request->status,
            ]);
            
            $menuData = [];
            for ($i = 1; $i <= 26; $i++) {
                $menuData[] = [
                    'role_id' => $role->id,
                    'menu_id' => $i,
                    'view' => $request->input("view_$i", 0),
                    'created' => $request->input("created_$i", 0),
                    'edit' => $request->input("edited_$i", 0),
                    'deleted' => $request->input("deleted_$i", 0),
                ];
            }
    
            // Insert or Update Menu Permissions
            foreach ($menuData as $data) {
                DB::table('tbl_role_permissions')->updateOrInsert(
                    ['role_id' => $data['role_id'], 'menu_id' => $data['menu_id']],
                    $data
                );
            }
            
            DB::commit();
            return redirect()->route('roles.index')->with('success','Role Updated Successfully.');

        } catch (\Throwable $th) {
            // Rollback and return with Error
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }

    /**
     * Delete User
     * @param User $user
     * @return Index Users
     * @author Rakesh Kumar
     */
    public function delete(Role $role)
    {

        DB::beginTransaction();
        try {
            // Delete User
            Role::whereId($role->id)->delete();
            DB::table('tbl_role_permissions')->where('role_id', $role->id)->delete();
            DB::commit();
            return redirect()->route('roles.index')->with('success', 'Role Deleted Successfully!.');

        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
