<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use App\Models\HectareMaster;
use App\Models\HectareMeasurement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class HectareMasterController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * List User
     * @param Nill
     * @return Array $user
     * @author Rakesh Kumar
     */
    public function index()
    {
        $hectareMaster = HectareMaster::select([
                'hectare_masters.*',
                'role.role_name',
                'user.first_name',
                'user.last_name',
            ])
            ->leftJoin('roles as role','role.id','=','hectare_masters.role_id')
            ->leftJoin('users as user','user.id','=','hectare_masters.user_id')
            ->paginate(10);
        return view('hectareMaster.index', ['hectareMaster' => $hectareMaster]);
    }

    /**
     * Create User
     * @param Nill
     * @return Array $user
     * @author Rakesh Kumar
     */
    public function create()
    {
        $roles = Role::select(['roles.id','roles.role_name','roles.status'])->where('status',1)->get();
        $users = User::select(['users.id','users.first_name','users.last_name','users.status'])->where('status',1)->get();
        return view('hectareMaster.add',['roles' => $roles, 'users' => $users]);
    }

    /**
     * Store User
     * @param Request $request
     * @return View Users
     * @author Rakesh Kumar
     */
    public function store(Request $request)
    {
        $userId = Auth::user()->id;
        $request->validate([
            'role_id' => 'required',
        ]);
        
        DB::beginTransaction();
        try {
            $hectareMaster = HectareMaster::create([
                'role_id' => $request->role_id,
                'user_id' => $request->user_id,
                'status' => $request->status,
            ]);
            
            foreach ($request->hectare as $hectare) {
                HectareMeasurement::create([
                    'hectare_id' => $hectareMaster->id,
                    'user_id' => $request->user_id,
                    'value' => $hectare['value'],
                    'unit' => $hectare['unit'],
                ]);
            }
            
            DB::commit();
            return redirect()->route('hectareMaster.index')->with('success', 'Hectare master Created Successfully.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }
    
    public function view($id)
    {
        $hectareMaster = HectareMaster::select([
            'hectare_masters.*',
            'roles.role_name as role_name',
            'users.first_name',
            'users.last_name',
        ])
        ->leftJoin('roles','roles.id','=','hectare_masters.role_id')
        ->leftJoin('users','users.id','=','hectare_masters.user_id')
        ->find($id);
            
        $hectareMeasurement = HectareMeasurement::where('hectare_id',$id)->get();
        
        return view('hectareMaster.view',[
            'hectareMaster'         => $hectareMaster,
            'hectareMeasurement'    => $hectareMeasurement
        ]);
    }
    
    public function getUsersByRoleId($role_id)
    {
        $users = User::where('role_id', $role_id)->get();
        return response()->json(['users' => $users]);
    }

    /**
     * Update Status Of User
     * @param Integer $status
     * @return List Page With Success
     * @author Rakesh Kumar
     */
    public function updateStatus($id, $status)
    {
        $validate = Validator::make([
            'id'        => $id,
            'status'    => $status
        ], 
        [
            'id'        =>  'required|exists:hectare_masters,id',
            'status'    =>  'required|in:0,1',
        ]);
    
        if ($validate->fails()) {
            return redirect()->route('hectareMaster.index')->with('error', $validate->errors()->first());
        }
        
        try {
            DB::beginTransaction();
            // Update status
            HectareMaster::whereId($id)->update([
                'status' => $status
            ]);
    
            DB::commit();
            return redirect()->route('hectareMaster.index')->with('success', 'Hectare Master Status Updated Successfully!');
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    
    /**
     * Edit Role
     * @param Integer $role
     * @return Collection $role
     * @author Rakesh Kumar
     */
    public function edit(Request $request, $id)
    {
        $hectareMaster = HectareMaster::find($id);
        $hectareMeasurement = HectareMeasurement::where('hectare_id',$id)->get();
        $roles = Role::select(['roles.id','roles.role_name','roles.status'])->where('status',1)->get();
        $users = User::select(['users.id','users.first_name','users.last_name','users.status'])->where('status',1)->get();
        return view('hectareMaster.edit', [
            'hectareMaster' => $hectareMaster,
            'hectareMeasurement' => $hectareMeasurement,
            'roles' => $roles, 
            'users' => $users
        ]);
    }

    /**
     * Update Role
     * @param Request $request, User $user
     * @return View Users
     * @author Rakesh Kumar
     */
    public function update(Request $request, $hectare_id)
    {
        $userId = Auth::user()->id;
        $request->validate([
            'role_id' => 'required',
        ]);
       
        
        DB::beginTransaction();
        try {
            $hectareMaster = HectareMaster::findOrFail($hectare_id);
            $hectareMeasturementIds = HectareMeasurement::where('hectare_id',$hectare_id)->pluck('id');
            HectareMeasurement::whereIn('id', $hectareMeasturementIds)->update([
                'user_id' => $request->user_id
            ]);
            $hectare['hectare'] = [];
            if(isset($request['hectare']) && $request['hectare'])
            {
               // $request['hectare'] = $request['hectare'];
                $hectare['hectare'] =  $request['hectare'];
            }
            $hectareMaster->update([
                'role_id' => $request->role_id,
                'user_id' => $request->user_id,
                'status' => $request->status,
            ]);
           

           
           if(!empty($request['hectare']))	{
		       
    		       foreach($hectare['hectare'] as $k=>$val){
    		            $hectare['hectare'][$k]['user_id'] =  $request->user_id;
    		       }
    		      // dd($hectare['hectare']);
		     
				 self::handleHectare($hectareMaster->id, $hectare['hectare']);
			}
			
            DB::commit();
            return redirect()->route('hectareMaster.index')->with('success','Hectare Master Updated Successfully.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', $th->getMessage());
        }
    }
    
    public static function handleHectare($id, $hactareData)
    {
        HectareMeasurement::where('hectare_id', $id)->delete();
       
        foreach ($hactareData as $hD) 
        {
            $relation = new HectareMeasurement();
            $relation->hectare_id       = $id;
            $relation->user_id       =  $hD['user_id'];
            $relation->value            = $hD['value'];
            $relation->unit             = $hD['unit'];
            $relation->save();
        }
    }

    /**
     * Delete User
     * @param User $user
     * @return Index Users
     * @author Rakesh Kumar
     */
    public function delete($id)
    {
        DB::beginTransaction();
        try {
            // Delete User
            HectareMaster::whereId($id)->delete();
            DB::commit();
            return redirect()->route('hectareMaster.index')->with('success', 'Hectare Master Deleted Successfully!.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
